﻿IF EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[hips].[PatientMasterGetByIhi]') AND type in (N'P', N'PC'))
BEGIN
	DROP PROCEDURE [hips].[PatientMasterGetByIhi]
END
GO
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
-- =============================================
-- Create date:  23 February 2016
-- Description:  Gets all Patient Master records for a specified IHI.
--
-- The accessing organisation must be specified using at least one of:
--   1. HpiO
--   2. HospitalId
--   3. HealthProviderOrganisationId
--   4. HealthProviderOrganisationNetworkId
-- =============================================
CREATE PROCEDURE [hips].[PatientMasterGetByIhi] 
(
	@Ihi VARCHAR(16) = NULL,
	@HpiO VARCHAR(16) = NULL,
	@HospitalId INT = NULL,
	@HealthProviderOrganisationId INT = NULL,
	@HealthProviderOrganisationNetworkId INT = NULL
)
AS
BEGIN
	IF @HealthProviderOrganisationNetworkId IS NULL AND @HospitalId IS NULL AND @HealthProviderOrganisationId IS NULL AND @HpiO IS NULL
	BEGIN
		RAISERROR(50003, -1, -1) 
	END
	
	SELECT pm.[PatientMasterId]
			, COALESCE (pmi.RegisteredSexId,pm.CurrentSexId, -1) AS RegisteredSexId
			, COALESCE (pm.CurrentSexId, -1) AS CurrentSexId
			, pmi.Ihi
			, COALESCE (pmi.IhiStatusId, -1) AS IhiStatusId
			, ihis.Description AS IhiStatus
			, COALESCE (pmi.IhiRecordStatusId, -1) AS IhiRecordStatusId
			, irs.Description AS IhiRecordStatus
			, pmi.[RegisteredFamilyName]
			, pmi.[RegisteredGivenName]
			, pmi.DateLastValidated AS IhiLastValidated
			, pm.[MedicareNumber]
			, pm.[MedicareIrn]
			, pm.[IsMedicareNumberValid]
			, pm.[DvaNumber]
			, pm.[StatePatientId]
			, pm.[DateOfBirth]
			, COALESCE (pmi.RegisteredDateOfBirth, pm.DateOfBirth) AS RegisteredDateOfBirth
			, hpo.[HealthProviderOrganisationNetworkId]
			, pm.[DateOfDeath]
			, pm.[DeathIndicatorId]
			, pm.[DateCreated]
			, pm.[UserCreated]
			, pm.[DateModified]
			, pm.[UserModified]         
	    FROM       hips.PatientMaster AS pm
        INNER JOIN hips.HospitalPatient AS hp
			ON pm.PatientMasterId = hp.PatientMasterId
		INNER JOIN hips.Hospital AS h
			ON hp.HospitalId = h.HospitalId
		INNER JOIN hips.HealthProviderOrganisation AS hpo
			ON h.HealthProviderOrganisationId = hpo.HealthProviderOrganisationId	     
        INNER JOIN [hips].[PatientMasterIhi] AS pmi 
			ON pmi.PatientMasterId = pm.PatientMasterId
			AND pmi.HealthProviderOrganisationNetworkId = hpo.HealthProviderOrganisationNetworkId
        INNER JOIN [hips].IhiRecordStatus AS irs 
			ON irs.IhiRecordStatusId = pmi.IhiRecordStatusId
		INNER JOIN [hips].IhiStatus AS ihis 
			ON ihis.IhiStatusId = pmi.IhiStatusId

    WHERE pmi.Ihi = @Ihi
		AND (@HospitalId IS NULL OR h.HospitalId = @HospitalId)
		AND (@HealthProviderOrganisationId IS NULL OR hpo.HealthProviderOrganisationId = @HealthProviderOrganisationId)
		AND (@HealthProviderOrganisationNetworkId IS NULL OR hpo.HealthProviderOrganisationNetworkId = @HealthProviderOrganisationNetworkId)
		AND (@HpiO IS NULL OR hpo.HpiO = @HpiO)

END
GO
GRANT EXEC ON [hips].[PatientMasterGetByIhi] TO RunStoredProcedure
GO
IF EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[hips].[PatientMasterGetByDemographics]') AND type in (N'P', N'PC'))
BEGIN
	DROP PROCEDURE [hips].[PatientMasterGetByDemographics]
END
GO
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
-- =============================================
-- Create date:   23 February 2016
-- Description:   Gets all Patient Master records that match the given demographics.
--
-- Lookup Types Supported:
--    1. Demographic search using all of:
--         FamilyName, GivenNames, DateOfBirth, CurrentSexId
--       and optionally DvaNumber
--       and optionally MedicareNumber, with or without MedicareIrn.
--
-- The business logic must ensure that special Medicare card values 
-- (invalid/ineligible) are passed in as null to avoid matching them.
-- Any IRN will match a null IRN in the record.
--
-- The @GivenNames parameter can include one or more given names, separated by spaces.
-- In order to match an existing patient record, the first or second given name that is 
-- provided must be one of the current given names of the patient record. For example,
-- parameter "ANNA BETTY CINDY" matches stored "DELILA ETHEL BETTY" because "BETTY" is
-- one of the first two names provided. However, that same parameter would not match
-- an existing patient "CINDY FREDA" because "CINDY" was not one of the first two names.
--
-- The accessing organisation must be specified using at least one of:
--   1. HpiO
--   2. HospitalId
--   3. HealthProviderOrganisationId
--   4. HealthProviderOrganisationNetworkId
-- 
-- Unlike the original hips.PatientMasterGet stored procedure, this procedure
-- allows P2P receive to look up patient by demographics, with the organisation
-- specified by HPI-O number, providing one or more of the given names (which 
-- must be in the right order) and does not require Medicare or DVA number.
-- =============================================
CREATE PROCEDURE hips.PatientMasterGetByDemographics 
(
	@FamilyName VARCHAR(80) = NULL,
	@GivenNames VARCHAR(80) = NULL,
	@DateOfBirth DATETIME = NULL,
	@CurrentSexId INT = NULL,
	@MedicareNumber VARCHAR(12) = NULL,
	@MedicareIrn VARCHAR(3) = NULL,
	@DvaNumber VARCHAR(18) = NULL,
	@HpiO VARCHAR(16) = NULL,
	@HospitalId INT = NULL,
	@HealthProviderOrganisationId INT = NULL,
	@HealthProviderOrganisationNetworkId INT = NULL
)
AS
BEGIN
	-- Accessing Organisation must be specified using at least one of these 4 options.
	IF @HealthProviderOrganisationNetworkId IS NULL AND @HospitalId IS NULL AND @HealthProviderOrganisationId IS NULL AND @HpiO IS NULL
	BEGIN
		RAISERROR(50003, -1, -1) 
	END

	SET @GivenNames = LTRIM(@GivenNames)
	DECLARE @FirstSpace INT = CHARINDEX(' ', @GivenNames + ' ')
	DECLARE @SecondSpace INT = CHARINDEX(' ', @GivenNames + '  ', @FirstSpace + 1)
	DECLARE @FirstName VARCHAR(80) = LEFT(@GivenNames, @FirstSpace - 1)
	DECLARE @SecondName VARCHAR(80) = SUBSTRING(@GivenNames, @FirstSpace + 1, @SecondSpace - @FirstSpace - 1)
	
	SELECT pm.PatientMasterId
			, COALESCE (pmi.RegisteredSexId,pm.CurrentSexId, -1) AS RegisteredSexId
			, COALESCE (pm.CurrentSexId, -1) AS CurrentSexId
			, pmi.Ihi
			, COALESCE (pmi.IhiStatusId, -1) AS IhiStatusId
			, ihis.Description AS IhiStatus
			, COALESCE (pmi.IhiRecordStatusId, -1) AS IhiRecordStatusId
			, irs.Description AS IhiRecordStatus
			, pmi.RegisteredFamilyName
			, pmi.RegisteredGivenName
			, pmi.DateLastValidated AS IhiLastValidated
			, pm.MedicareNumber
			, pm.MedicareIrn
			, pm.IsMedicareNumberValid
			, pm.DvaNumber
			, pm.StatePatientId
			, pm.DateOfBirth
			, COALESCE (pmi.RegisteredDateOfBirth, pm.DateOfBirth) AS RegisteredDateOfBirth
			, hpo.HealthProviderOrganisationNetworkId
			, pm.DateOfDeath
			, pm.DeathIndicatorId
			, pm.DateCreated
			, pm.UserCreated
			, pm.DateModified
			, pm.UserModified
	FROM	hips.PatientMaster AS pm
	INNER JOIN hips.HospitalPatient AS hp 
		ON pm.PatientMasterId = hp.PatientMasterId
	INNER JOIN hips.Hospital AS h 
		ON h.HospitalId = hp.HospitalId
	INNER JOIN hips.HealthProviderOrganisation AS hpo
		ON h.HealthProviderOrganisationId = hpo.HealthProviderOrganisationId	
	INNER JOIN hips.PatientMasterName pmn 
		ON pmn.PatientMasterId = pm.PatientMasterId
		AND pmn.NameTypeId = 2
	LEFT JOIN 
		(
			hips.PatientMasterIhi AS pmi 
			INNER JOIN hips.IhiRecordStatus AS irs 
				ON irs.IhiRecordStatusId = pmi.IhiRecordStatusId
			INNER JOIN hips.IhiStatus AS ihis 
				ON ihis.IhiStatusId = pmi.IhiStatusId
		)
		ON pmi.PatientMasterID = pm.PatientMasterID
		AND pmi.HealthProviderOrganisationNetworkId = hpo.HealthProviderOrganisationNetworkId
		
	WHERE	(pmn.FamilyName = @FamilyName)
	AND		(pm.DateOfBirth = @DateOfBirth)
	AND		(pm.CurrentSexId = @CurrentSexId)
	AND		(@DvaNumber IS NULL OR pm.DvaNumber = @DvaNumber)
	AND		(@MedicareNumber IS NULL OR pm.MedicareNumber = @MedicareNumber)
	AND		(@MedicareIrn IS NULL OR pm.MedicareIrn IS NULL OR pm.MedicareIrn = @MedicareIrn)
	AND		(' ' + pmn.GivenNames + ' ' LIKE '% ' + @FirstName + ' %'
		OR	 ' ' + pmn.GivenNames + ' ' LIKE '% ' + @SecondName + ' %')
	AND		(@HospitalId IS NULL OR h.HospitalId = @HospitalId)
	AND		(@HealthProviderOrganisationId IS NULL OR hpo.HealthProviderOrganisationId = @HealthProviderOrganisationId)
	AND		(@HealthProviderOrganisationNetworkId IS NULL OR hpo.HealthProviderOrganisationNetworkId = @HealthProviderOrganisationNetworkId)
	AND		(@HpiO IS NULL OR hpo.HpiO = @HpiO)
	
END
GO
GRANT EXEC ON [hips].[PatientMasterGetByDemographics] TO RunStoredProcedure
GO